# Landing Page Builder - Sistema Completo de Criação de Landing Pages

## Visão Geral

O Landing Page Builder é um sistema full-stack completo desenvolvido para criar landing pages de alta conversão de forma rápida e eficiente. O sistema combina um backend robusto em Laravel com um frontend moderno em React, oferecendo uma experiência de usuário intuitiva e recursos avançados para criação, edição e gerenciamento de landing pages.

### Características Principais

- **Arquitetura Moderna**: Backend Laravel 11 com API REST e frontend React com Next.js
- **Sistema Modular**: Componentes reutilizáveis para diferentes seções de landing pages
- **Editor Visual**: Interface intuitiva para criação e edição de páginas
- **Testes A/B**: Funcionalidade integrada para otimização de conversões
- **Responsivo**: Design adaptável para desktop, tablet e mobile
- **Autenticação JWT**: Sistema seguro de autenticação para administradores
- **Templates Dinâmicos**: Sistema flexível de templates e seções modulares

## Arquitetura do Sistema

### Backend (Laravel 11)

O backend funciona como um CMS headless, fornecendo uma API REST completa para gerenciamento de landing pages, templates e conteúdo.

#### Principais Componentes:

- **Modelos de Dados**: User, LandingPage, Template, TemplateSection, PageContent
- **Autenticação JWT**: Sistema seguro para proteção das rotas administrativas
- **API REST**: Endpoints completos para CRUD de páginas e templates
- **Sanitização HTML**: Proteção contra XSS com HTMLPurifier
- **Versionamento**: Sistema de rascunho/publicado e variantes para testes A/B

#### Estrutura do Banco de Dados:

- `users`: Gerenciamento de usuários administradores
- `templates`: Templates base para criação de páginas
- `template_sections`: Seções modulares dos templates
- `landing_pages`: Páginas criadas pelos usuários
- `page_content`: Conteúdo específico de cada seção das páginas

### Frontend (React + Next.js)

O frontend oferece uma interface moderna e responsiva para visualização das landing pages públicas.

#### Principais Componentes:

- **Roteamento Dinâmico**: Sistema que renderiza páginas baseado na URL
- **Componentes Modulares**: HeroSection, BenefitsSection, TestimonialsSection, CTASection, FooterSection
- **Integração API**: Biblioteca axios para comunicação com o backend
- **Design System**: Paleta de cores personalizada baseada em azul petróleo, branco, azul claro, preto e laranja
- **Responsividade**: Layout adaptável usando Tailwind CSS

## Instalação e Configuração

### Pré-requisitos

- PHP 8.1 ou superior
- Composer
- Node.js 18 ou superior
- SQLite (ou MySQL/PostgreSQL)

### Instalação do Backend (Laravel)

1. **Clone o repositório e navegue para o diretório do backend:**
   ```bash
   cd landing-page-builder
   ```

2. **Instale as dependências do PHP:**
   ```bash
   composer install
   ```

3. **Configure o arquivo de ambiente:**
   ```bash
   cp .env.example .env
   ```
   
   Edite o arquivo `.env` com suas configurações de banco de dados e outras variáveis necessárias.

4. **Gere a chave da aplicação:**
   ```bash
   php artisan key:generate
   ```

5. **Configure o JWT:**
   ```bash
   php artisan jwt:secret
   ```

6. **Execute as migrações:**
   ```bash
   php artisan migrate
   ```

7. **Execute os seeders para criar dados iniciais:**
   ```bash
   php artisan db:seed --class=DefaultTemplateSeeder
   ```

8. **Inicie o servidor de desenvolvimento:**
   ```bash
   php artisan serve --host=0.0.0.0 --port=8000
   ```

### Instalação do Frontend (React)

1. **Navegue para o diretório do frontend:**
   ```bash
   cd ../landing-page-frontend
   ```

2. **Instale as dependências:**
   ```bash
   pnpm install
   ```

3. **Configure as variáveis de ambiente:**
   Crie um arquivo `.env.local` com:
   ```
   VITE_API_URL=http://localhost:8000/api
   ```

4. **Inicie o servidor de desenvolvimento:**
   ```bash
   pnpm run dev
   ```

## Estrutura de Arquivos

### Backend (Laravel)
```
landing-page-builder/
├── app/
│   ├── Http/
│   │   └── Controllers/
│   │       ├── AuthController.php
│   │       ├── LandingPageController.php
│   │       └── TemplateController.php
│   └── Models/
│       ├── User.php
│       ├── LandingPage.php
│       ├── Template.php
│       ├── TemplateSection.php
│       └── PageContent.php
├── database/
│   ├── migrations/
│   └── seeders/
├── routes/
│   └── api.php
└── config/
```

### Frontend (React)
```
landing-page-frontend/
├── src/
│   ├── components/
│   │   ├── LandingPageRenderer.jsx
│   │   └── sections/
│   │       ├── HeroSection.jsx
│   │       ├── BenefitsSection.jsx
│   │       ├── TestimonialsSection.jsx
│   │       ├── CTASection.jsx
│   │       └── FooterSection.jsx
│   ├── lib/
│   │   └── api.js
│   ├── App.jsx
│   └── App.css
├── public/
└── package.json
```


## API Endpoints

### Autenticação

| Método | Endpoint | Descrição |
|--------|----------|-----------|
| POST | `/api/auth/login` | Autenticação de usuário |
| POST | `/api/auth/logout` | Logout do usuário |
| POST | `/api/auth/refresh` | Renovar token JWT |
| POST | `/api/auth/me` | Obter dados do usuário autenticado |

### Templates

| Método | Endpoint | Descrição |
|--------|----------|-----------|
| GET | `/api/templates` | Listar todos os templates |
| GET | `/api/templates/{id}` | Obter template específico |
| POST | `/api/admin/templates` | Criar novo template |
| PUT | `/api/admin/templates/{id}` | Atualizar template |
| DELETE | `/api/admin/templates/{id}` | Excluir template |

### Landing Pages

| Método | Endpoint | Descrição |
|--------|----------|-----------|
| GET | `/api/admin/pages` | Listar páginas do usuário |
| GET | `/api/admin/pages/{id}` | Obter página específica |
| POST | `/api/admin/pages` | Criar nova página |
| PUT | `/api/admin/pages/{id}` | Atualizar página |
| DELETE | `/api/admin/pages/{id}` | Excluir página |
| POST | `/api/admin/pages/{id}/publish` | Publicar página |
| POST | `/api/admin/pages/{id}/create-variant` | Criar variante para teste A/B |

### Páginas Públicas

| Método | Endpoint | Descrição |
|--------|----------|-----------|
| GET | `/api/public/page/{slug}` | Obter página pública por slug |

## Uso do Sistema

### Criação de Landing Pages

1. **Seleção de Template**: Escolha um template base ou use o template "Blocos de Construção"
2. **Configuração Básica**: Defina título, slug e meta informações
3. **Edição de Conteúdo**: Customize cada seção com seu conteúdo específico
4. **Preview**: Visualize a página antes da publicação
5. **Publicação**: Torne a página disponível publicamente

### Componentes Modulares Disponíveis

#### Hero Section
- Título principal impactante
- Subtítulo descritivo
- Botão de chamada para ação
- Imagem de destaque
- Layout flexível (centralizado ou lateral)

#### Benefits Section
- Lista de benefícios com ícones
- Layout em grid responsivo
- Título da seção customizável
- Suporte a múltiplos itens

#### Testimonials Section
- Depoimentos de clientes
- Sistema de avaliação com estrelas
- Layout em cards
- Informações do autor

#### CTA Section
- Chamada para ação final
- Design impactante com gradiente
- Botão de destaque
- Texto persuasivo

#### Footer Section
- Informações de copyright
- Links importantes
- Layout responsivo
- Customização completa

### Testes A/B

O sistema permite criar variantes de páginas para otimização de conversões:

1. **Criar Variante**: Gere uma versão alternativa da página principal
2. **Personalizar**: Modifique elementos específicos na variante
3. **Publicar**: Disponibilize ambas as versões
4. **Monitorar**: Acompanhe métricas de conversão
5. **Otimizar**: Use os dados para melhorar performance

## Personalização e Extensão

### Adicionando Novos Componentes

Para adicionar um novo tipo de seção:

1. **Backend**: Crie uma nova entrada no seeder com o tipo de seção
2. **Frontend**: Desenvolva o componente React correspondente
3. **Integração**: Adicione o componente ao LandingPageRenderer
4. **Teste**: Verifique a funcionalidade completa

### Customização de Estilos

O sistema usa um design system baseado em CSS custom properties:

```css
:root {
  --primary: oklch(0.37 0.05 200); /* Azul Petróleo */
  --secondary: oklch(0.75 0.05 220); /* Azul Claro */
  --accent: oklch(0.7 0.15 45); /* Laranja */
}
```

### Integração com Ferramentas Externas

O sistema está preparado para integração com:

- **Google Analytics**: Rastreamento de conversões
- **WooCommerce**: Redirecionamento para checkout
- **Facebook Pixel**: Tracking de eventos
- **Ferramentas de Email Marketing**: Captura de leads

## Deploy em Produção

### Configuração do Servidor

1. **Requisitos do Servidor**:
   - PHP 8.1+ com extensões necessárias
   - Composer instalado
   - Node.js para build do frontend
   - Servidor web (Apache/Nginx)
   - Banco de dados (MySQL/PostgreSQL/SQLite)

2. **Configuração do Laravel**:
   ```bash
   composer install --optimize-autoloader --no-dev
   php artisan config:cache
   php artisan route:cache
   php artisan view:cache
   ```

3. **Build do Frontend**:
   ```bash
   pnpm run build
   ```

### Deploy no cPanel

1. **Upload dos Arquivos**:
   - Faça upload do backend para a pasta raiz
   - Coloque os arquivos do build do frontend em `public_html`

2. **Configuração do Banco**:
   - Crie o banco de dados no painel
   - Configure as credenciais no `.env`
   - Execute as migrações

3. **Configuração de Domínio**:
   - Configure o domínio principal para o frontend
   - Configure um subdomínio para a API (ex: api.seudominio.com)

### Configuração para napoleon.com.br

Para instalação no domínio especificado:

1. **Subdomínio API**: Configure `api.napoleon.com.br` para o backend Laravel
2. **Domínio Principal**: Use `napoleon.com.br` para o frontend React
3. **SSL**: Configure certificados SSL para ambos os domínios
4. **CORS**: Configure adequadamente as origens permitidas no backend

## Segurança

### Medidas Implementadas

- **Autenticação JWT**: Tokens seguros com expiração configurável
- **Sanitização HTML**: Proteção contra ataques XSS
- **Validação de Dados**: Validação rigorosa em todas as entradas
- **CORS Configurado**: Controle de acesso entre domínios
- **Rate Limiting**: Proteção contra ataques de força bruta

### Recomendações Adicionais

- Configure HTTPS em produção
- Use senhas fortes para usuários administrativos
- Mantenha as dependências atualizadas
- Configure backups regulares do banco de dados
- Monitore logs de acesso e erro

## Manutenção e Suporte

### Logs e Monitoramento

- **Laravel Logs**: Disponíveis em `storage/logs/`
- **Métricas de Performance**: Monitore tempo de resposta da API
- **Análise de Conversão**: Acompanhe métricas das landing pages

### Atualizações

- **Backend**: Use `composer update` para atualizar dependências
- **Frontend**: Use `pnpm update` para atualizar pacotes
- **Migrações**: Execute `php artisan migrate` após atualizações

### Backup

Realize backups regulares de:
- Banco de dados
- Arquivos de configuração
- Uploads de usuários
- Logs importantes

## Conclusão

O Landing Page Builder oferece uma solução completa e moderna para criação de landing pages de alta conversão. Com sua arquitetura modular e interface intuitiva, o sistema permite criar páginas profissionais rapidamente, mantendo a flexibilidade para customizações avançadas.

O sistema está pronto para produção e pode ser facilmente expandido com novos recursos e integrações conforme necessário. A documentação completa e a estrutura bem organizada facilitam a manutenção e evolução contínua da plataforma.

