<?php

namespace App\Http\Controllers;

use App\Models\LandingPage;
use App\Models\PageContent;
use App\Models\Template;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class LandingPageController extends Controller
{
    public function __construct()
    {
        $this->middleware(
            'auth:api',
            ['except' => ['showPublicPage']]
        );
    }

    /**
     * Display a listing of the main landing pages.
     */
    public function index()
    {
        $pages = Auth::user()->landingPages()->main()->with('template')->get();
        return response()->json($pages);
    }

    /**
     * Store a newly created landing page in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'template_id' => 'required|exists:templates,id',
        ]);

        return DB::transaction(function () use ($request) {
            $template = Template::findOrFail($request->template_id);

            $page = new LandingPage([
                'user_id' => Auth::id(),
                'template_id' => $template->id,
                'title' => $request->title,
                'slug' => Str::slug($request->title) . '-' . Str::random(5),
                'status' => 'draft',
                'version_type' => 'main',
            ]);
            $page->save();

            foreach ($template->sections as $section) {
                PageContent::create([
                    'landing_page_id' => $page->id,
                    'section_type' => $section->section_type,
                    'section_name' => $section->section_name,
                    'content_data' => $section->default_content,
                    'sort_order' => $section->sort_order,
                    'is_visible' => true,
                ]);
            }

            return response()->json($page->load('content'), 201);
        });
    }

    /**
     * Display the specified landing page.
     */
    public function show(LandingPage $landingPage)
    {
        if ($landingPage->user_id !== Auth::id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
        return response()->json($landingPage->load('content', 'variants'));
    }

    /**
     * Update the specified landing page in storage.
     */
    public function update(Request $request, LandingPage $landingPage)
    {
        if ($landingPage->user_id !== Auth::id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'slug' => 'sometimes|required|string|max:255|unique:landing_pages,slug,' . $landingPage->id,
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'custom_html' => 'nullable|string',
            'content' => 'sometimes|array',
        ]);

        $landingPage->fill($request->only([
            'title', 'slug', 'meta_title', 'meta_description', 'custom_html'
        ]));

        if ($request->has('custom_html')) {
            $landingPage->custom_html = Purifier::clean($request->custom_html); // Sanitize HTML
        }

        if ($request->has('content')) {
            foreach ($request->content as $sectionData) {
                PageContent::updateOrCreate(
                    [
                        'landing_page_id' => $landingPage->id,
                        'section_type' => $sectionData['section_type'],
                        'section_name' => $sectionData['section_name'],
                    ],
                    [
                        'content_data' => $sectionData['content_data'],
                        'sort_order' => $sectionData['sort_order'] ?? 0,
                        'is_visible' => $sectionData['is_visible'] ?? true,
                    ]
                );
            }
        }

        $landingPage->save();

        return response()->json($landingPage->load('content'));
    }

    /**
     * Publish the specified landing page.
     */
    public function publish(LandingPage $landingPage)
    {
        if ($landingPage->user_id !== Auth::id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $landingPage->status = 'published';
        $landingPage->save();

        return response()->json(['message' => 'Landing page published successfully.', 'page' => $landingPage]);
    }

    /**
     * Create a variant for the specified landing page.
     */
    public function createVariant(LandingPage $landingPage)
    {
        if ($landingPage->user_id !== Auth::id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        if ($landingPage->version_type !== 'main') {
            return response()->json(['message' => 'Only main pages can have variants.'], 400);
        }

        return DB::transaction(function () use ($landingPage) {
            $variant = $landingPage->replicate();
            $variant->slug = $landingPage->slug . '-variant-' . Str::random(5);
            $variant->version_type = 'variant_b';
            $variant->main_page_id = $landingPage->id;
            $variant->status = 'draft';
            $variant->title = $landingPage->title . ' (Variant B)';
            $variant->save();

            foreach ($landingPage->content as $content) {
                PageContent::create([
                    'landing_page_id' => $variant->id,
                    'section_type' => $content->section_type,
                    'section_name' => $content->section_name,
                    'content_data' => $content->content_data,
                    'sort_order' => $content->sort_order,
                    'is_visible' => $content->is_visible,
                ]);
            }

            return response()->json($variant->load('content'), 201);
        });
    }

    /**
     * Display the public landing page.
     */
    public function showPublicPage(Request $request, $slug)
    {
        $page = LandingPage::where('slug', $slug)->published()->firstOrFail();

        // Increment views (optional, depending on analytics setup)
        $page->incrementViews();

        // Add ETag for caching
        $eTag = md5($page->updated_at . $page->status . $page->custom_html . json_encode($page->content));
        if ($request->header('If-None-Match') === $eTag) {
            return response('', 304);
        }

        return response()->json($page->load('content'))->withHeaders([
            'ETag' => $eTag,
            'Cache-Control' => 'public, max-age=300, must-revalidate',
        ]);
    }

    /**
     * Remove the specified landing page from storage.
     */
    public function destroy(LandingPage $landingPage)
    {
        if ($landingPage->user_id !== Auth::id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $landingPage->delete();

        return response()->json(['message' => 'Landing page deleted successfully.']);
    }
}

