<?php

namespace App\Http\Controllers;

use App\Models\Template;
use App\Models\TemplateSection;
use Illuminate\Http\Request;

class TemplateController extends Controller
{
    public function __construct()
    {
        $this->middleware(
            'auth:api',
            ['except' => ['index']]
        );
    }

    /**
     * Display a listing of the templates.
     */
    public function index()
    {
        $templates = Template::active()->with('sections')->get();
        return response()->json($templates);
    }

    /**
     * Store a newly created template in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'nullable|string',
            'preview_image' => 'nullable|string',
            'sections' => 'required|array',
            'sections.*.section_type' => 'required|string',
            'sections.*.section_name' => 'required|string',
            'sections.*.default_content' => 'required|array',
            'sections.*.sort_order' => 'nullable|integer',
            'sections.*.is_required' => 'nullable|boolean',
        ]);

        $template = Template::create($request->only('name', 'description', 'category', 'preview_image'));

        foreach ($request->sections as $sectionData) {
            $template->sections()->create($sectionData);
        }

        return response()->json($template->load('sections'), 201);
    }

    /**
     * Display the specified template.
     */
    public function show(Template $template)
    {
        return response()->json($template->load('sections'));
    }

    /**
     * Update the specified template in storage.
     */
    public function update(Request $request, Template $template)
    {
        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'nullable|string',
            'preview_image' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'sections' => 'sometimes|array',
            'sections.*.id' => 'sometimes|required|exists:template_sections,id',
            'sections.*.section_type' => 'required|string',
            'sections.*.section_name' => 'required|string',
            'sections.*.default_content' => 'required|array',
            'sections.*.sort_order' => 'nullable|integer',
            'sections.*.is_required' => 'nullable|boolean',
        ]);

        $template->update($request->only('name', 'description', 'category', 'preview_image', 'is_active'));

        if ($request->has('sections')) {
            foreach ($request->sections as $sectionData) {
                if (isset($sectionData['id'])) {
                    $template->sections()->where('id', $sectionData['id'])->update($sectionData);
                } else {
                    $template->sections()->create($sectionData);
                }
            }
        }

        return response()->json($template->load('sections'));
    }

    /**
     * Remove the specified template from storage.
     */
    public function destroy(Template $template)
    {
        $template->delete();

        return response()->json(['message' => 'Template deleted successfully.']);
    }
}

