<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class LandingPage extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'template_id',
        'title',
        'slug',
        'meta_title',
        'meta_description',
        'status',
        'version_type',
        'main_page_id',
        'custom_html',
        'seo_settings',
        'analytics_settings',
        'views',
        'conversions',
    ];

    protected $casts = [
        'seo_settings' => 'array',
        'analytics_settings' => 'array',
    ];

    /**
     * Get the user that owns the landing page.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the template that the landing page uses.
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(Template::class);
    }

    /**
     * Get the main page if this is a variant.
     */
    public function mainPage(): BelongsTo
    {
        return $this->belongsTo(LandingPage::class, 'main_page_id');
    }

    /**
     * Get the variants of this page.
     */
    public function variants(): HasMany
    {
        return $this->hasMany(LandingPage::class, 'main_page_id');
    }

    /**
     * Get the page content sections.
     */
    public function content(): HasMany
    {
        return $this->hasMany(PageContent::class)->orderBy('sort_order');
    }

    /**
     * Scope a query to only include published pages.
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published');
    }

    /**
     * Scope a query to only include main pages (not variants).
     */
    public function scopeMain($query)
    {
        return $query->where('version_type', 'main');
    }

    /**
     * Increment the views counter.
     */
    public function incrementViews()
    {
        $this->increment('views');
    }

    /**
     * Increment the conversions counter.
     */
    public function incrementConversions()
    {
        $this->increment('conversions');
    }
}

