<?php
/**
 * Script de verificação de requisitos do sistema
 * Landing Page Builder - Installer
 */

header('Content-Type: application/json');

function checkSystemRequirements() {
    $requirements = [];
    
    // Verificar versão do PHP
    $requirements['php_version'] = [
        'name' => 'PHP 8.1 ou superior',
        'required' => '8.1.0',
        'current' => PHP_VERSION,
        'status' => version_compare(PHP_VERSION, '8.1.0', '>=')
    ];
    
    // Verificar extensões necessárias
    $extensions = [
        'pdo' => 'PDO',
        'json' => 'JSON',
        'mbstring' => 'Mbstring',
        'openssl' => 'OpenSSL',
        'tokenizer' => 'Tokenizer',
        'xml' => 'XML',
        'ctype' => 'Ctype',
        'fileinfo' => 'Fileinfo',
        'bcmath' => 'BCMath'
    ];
    
    foreach ($extensions as $ext => $name) {
        $requirements["ext_{$ext}"] = [
            'name' => "Extensão {$name}",
            'status' => extension_loaded($ext)
        ];
    }
    
    // Verificar permissões de diretórios
    $directories = [
        '../storage' => 'Diretório storage',
        '../bootstrap/cache' => 'Diretório bootstrap/cache',
        '../database' => 'Diretório database'
    ];
    
    foreach ($directories as $dir => $name) {
        $requirements["dir_" . str_replace(['/', '..'], ['_', 'parent'], $dir)] = [
            'name' => "{$name} (gravável)",
            'status' => is_dir($dir) && is_writable($dir)
        ];
    }
    
    // Verificar se o Composer foi executado
    $requirements['composer'] = [
        'name' => 'Dependências do Composer',
        'status' => file_exists('../vendor/autoload.php')
    ];
    
    // Verificar configuração do servidor web
    $requirements['mod_rewrite'] = [
        'name' => 'Mod Rewrite (Apache)',
        'status' => function_exists('apache_get_modules') ? in_array('mod_rewrite', apache_get_modules()) : true
    ];
    
    // Verificar limites do PHP
    $requirements['memory_limit'] = [
        'name' => 'Limite de memória (mín. 128M)',
        'current' => ini_get('memory_limit'),
        'status' => $this->checkMemoryLimit()
    ];
    
    $requirements['max_execution_time'] = [
        'name' => 'Tempo máximo de execução (mín. 60s)',
        'current' => ini_get('max_execution_time'),
        'status' => ini_get('max_execution_time') == 0 || ini_get('max_execution_time') >= 60
    ];
    
    // Verificar se já está instalado
    $requirements['not_installed'] = [
        'name' => 'Sistema não instalado',
        'status' => !file_exists('../.env') || !file_exists('../storage/installed.lock')
    ];
    
    return $requirements;
}

function checkMemoryLimit() {
    $memoryLimit = ini_get('memory_limit');
    
    if ($memoryLimit == -1) {
        return true; // Sem limite
    }
    
    $memoryLimit = $this->convertToBytes($memoryLimit);
    return $memoryLimit >= 128 * 1024 * 1024; // 128MB
}

function convertToBytes($value) {
    $value = trim($value);
    $last = strtolower($value[strlen($value)-1]);
    $value = (int) $value;
    
    switch($last) {
        case 'g':
            $value *= 1024;
        case 'm':
            $value *= 1024;
        case 'k':
            $value *= 1024;
    }
    
    return $value;
}

function getSystemInfo() {
    return [
        'php_version' => PHP_VERSION,
        'php_sapi' => PHP_SAPI,
        'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
        'document_root' => $_SERVER['DOCUMENT_ROOT'] ?? '',
        'current_user' => get_current_user(),
        'memory_limit' => ini_get('memory_limit'),
        'max_execution_time' => ini_get('max_execution_time'),
        'upload_max_filesize' => ini_get('upload_max_filesize'),
        'post_max_size' => ini_get('post_max_size'),
        'timezone' => date_default_timezone_get()
    ];
}

// Processar requisição
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $requirements = checkSystemRequirements();
    $systemInfo = getSystemInfo();
    
    $allOk = true;
    foreach ($requirements as $req) {
        if (!$req['status']) {
            $allOk = false;
            break;
        }
    }
    
    echo json_encode([
        'success' => $allOk,
        'requirements' => $requirements,
        'system_info' => $systemInfo,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
} else {
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
}
?>

