<?php
/**
 * Script de configuração do sistema
 * Landing Page Builder - Installer
 */

header('Content-Type: application/json');

class ConfigInstaller {
    
    public function createEnvFile($config) {
        try {
            $envContent = $this->generateEnvContent($config);
            
            // Backup do .env existente se houver
            if (file_exists('../.env')) {
                copy('../.env', '../.env.backup.' . date('Y-m-d-H-i-s'));
            }
            
            // Criar novo .env
            if (file_put_contents('../.env', $envContent) === false) {
                throw new Exception('Não foi possível criar o arquivo .env');
            }
            
            return [
                'success' => true,
                'message' => 'Arquivo de configuração criado com sucesso!'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao criar configuração: ' . $e->getMessage()
            ];
        }
    }
    
    private function generateEnvContent($config) {
        $appKey = $this->generateAppKey();
        $jwtSecret = $this->generateJWTSecret();
        
        $content = "APP_NAME=\"{$config['site_name']}\"
APP_ENV=production
APP_KEY={$appKey}
APP_DEBUG=false
APP_URL={$config['site_url']}

LOG_CHANNEL=stack
LOG_DEPRECATIONS_CHANNEL=null
LOG_LEVEL=debug

DB_CONNECTION={$config['db_type']}";

        if ($config['db_type'] === 'mysql') {
            $content .= "
DB_HOST={$config['db_host']}
DB_PORT={$config['db_port']}
DB_DATABASE={$config['db_name']}
DB_USERNAME={$config['db_user']}
DB_PASSWORD={$config['db_pass']}";
        } else {
            $content .= "
#DB_HOST=127.0.0.1
#DB_PORT=3306
#DB_DATABASE=laravel
#DB_USERNAME=root
#DB_PASSWORD=";
        }

        $content .= "

BROADCAST_DRIVER=log
CACHE_DRIVER=file
FILESYSTEM_DISK=local
QUEUE_CONNECTION=sync
SESSION_DRIVER=file
SESSION_LIFETIME=120

MEMCACHED_HOST=127.0.0.1

REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379

MAIL_MAILER=smtp
MAIL_HOST=mailpit
MAIL_PORT=1025
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
MAIL_FROM_ADDRESS=\"{$config['admin_email']}\"
MAIL_FROM_NAME=\"{$config['site_name']}\"

AWS_ACCESS_KEY_ID=
AWS_SECRET_ACCESS_KEY=
AWS_DEFAULT_REGION=us-east-1
AWS_BUCKET=
AWS_USE_PATH_STYLE_ENDPOINT=false

PUSHER_APP_ID=
PUSHER_APP_KEY=
PUSHER_APP_SECRET=
PUSHER_HOST=
PUSHER_PORT=443
PUSHER_SCHEME=https
PUSHER_APP_CLUSTER=mt1

VITE_APP_NAME=\"{$config['site_name']}\"
VITE_PUSHER_APP_KEY=\"\${PUSHER_APP_KEY}\"
VITE_PUSHER_HOST=\"\${PUSHER_HOST}\"
VITE_PUSHER_PORT=\"\${PUSHER_PORT}\"
VITE_PUSHER_SCHEME=\"\${PUSHER_SCHEME}\"
VITE_PUSHER_APP_CLUSTER=\"\${PUSHER_APP_CLUSTER}\"

JWT_SECRET={$jwtSecret}

# Landing Page Builder Specific
LPB_DOMAIN={$config['site_url']}
LPB_ADMIN_EMAIL={$config['admin_email']}
LPB_INSTALLED=true
LPB_VERSION=1.0.0";

        return $content;
    }
    
    private function generateAppKey() {
        return 'base64:' . base64_encode(random_bytes(32));
    }
    
    private function generateJWTSecret() {
        return base64_encode(random_bytes(64));
    }
    
    public function setPermissions() {
        try {
            $directories = [
                '../storage',
                '../storage/app',
                '../storage/framework',
                '../storage/framework/cache',
                '../storage/framework/sessions',
                '../storage/framework/views',
                '../storage/logs',
                '../bootstrap/cache'
            ];
            
            foreach ($directories as $dir) {
                if (!is_dir($dir)) {
                    mkdir($dir, 0755, true);
                }
                chmod($dir, 0755);
            }
            
            // Arquivos específicos
            $files = [
                '../database/database.sqlite'
            ];
            
            foreach ($files as $file) {
                if (file_exists($file)) {
                    chmod($file, 0664);
                }
            }
            
            return [
                'success' => true,
                'message' => 'Permissões configuradas com sucesso!'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao configurar permissões: ' . $e->getMessage()
            ];
        }
    }
    
    public function cacheConfig() {
        try {
            $output = [];
            $returnCode = 0;
            
            // Limpar cache existente
            exec('cd .. && php artisan config:clear 2>&1', $output, $returnCode);
            
            // Cachear configurações
            exec('cd .. && php artisan config:cache 2>&1', $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new Exception('Erro ao cachear configurações: ' . implode("\n", $output));
            }
            
            return [
                'success' => true,
                'message' => 'Configurações cacheadas com sucesso!',
                'output' => $output
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    public function createInstallLock() {
        try {
            $lockData = [
                'installed_at' => date('Y-m-d H:i:s'),
                'version' => '1.0.0',
                'installer_version' => '1.0.0',
                'php_version' => PHP_VERSION,
                'server' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'
            ];
            
            $lockContent = json_encode($lockData, JSON_PRETTY_PRINT);
            
            if (file_put_contents('../storage/installed.lock', $lockContent) === false) {
                throw new Exception('Não foi possível criar arquivo de bloqueio');
            }
            
            return [
                'success' => true,
                'message' => 'Instalação finalizada com sucesso!'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao finalizar instalação: ' . $e->getMessage()
            ];
        }
    }
    
    public function generateFrontendConfig($config) {
        try {
            $frontendConfig = [
                'VITE_API_URL' => rtrim($config['site_url'], '/') . '/api',
                'VITE_APP_NAME' => $config['site_name'],
                'VITE_APP_URL' => $config['site_url']
            ];
            
            $envContent = '';
            foreach ($frontendConfig as $key => $value) {
                $envContent .= "{$key}={$value}\n";
            }
            
            // Criar arquivo .env para o frontend se existir
            if (is_dir('../frontend')) {
                file_put_contents('../frontend/.env.production', $envContent);
            }
            
            return [
                'success' => true,
                'message' => 'Configuração do frontend criada!',
                'config' => $frontendConfig
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao configurar frontend: ' . $e->getMessage()
            ];
        }
    }
}

// Processar requisições
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $installer = new ConfigInstaller();
    
    switch ($action) {
        case 'create_env':
            echo json_encode($installer->createEnvFile($_POST));
            break;
            
        case 'set_permissions':
            echo json_encode($installer->setPermissions());
            break;
            
        case 'cache_config':
            echo json_encode($installer->cacheConfig());
            break;
            
        case 'create_lock':
            echo json_encode($installer->createInstallLock());
            break;
            
        case 'generate_frontend_config':
            echo json_encode($installer->generateFrontendConfig($_POST));
            break;
            
        default:
            echo json_encode([
                'success' => false,
                'message' => 'Ação inválida'
            ]);
    }
} else {
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
}
?>

