<?php
/**
 * Script de configuração e teste do banco de dados
 * Landing Page Builder - Installer
 */

header('Content-Type: application/json');

class DatabaseInstaller {
    
    public function testConnection($config) {
        try {
            $pdo = $this->createConnection($config);
            
            // Testar uma query simples
            $pdo->query('SELECT 1');
            
            return [
                'success' => true,
                'message' => 'Conexão estabelecida com sucesso!'
            ];
            
        } catch (PDOException $e) {
            return [
                'success' => false,
                'message' => 'Erro na conexão: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ];
        }
    }
    
    public function createDatabase($config) {
        if ($config['db_type'] === 'sqlite') {
            return $this->createSQLiteDatabase($config);
        } else {
            return $this->createMySQLDatabase($config);
        }
    }
    
    private function createSQLiteDatabase($config) {
        try {
            $dbPath = '../database/database.sqlite';
            
            // Criar diretório se não existir
            if (!is_dir('../database')) {
                mkdir('../database', 0755, true);
            }
            
            // Criar arquivo SQLite
            if (!file_exists($dbPath)) {
                touch($dbPath);
                chmod($dbPath, 0664);
            }
            
            // Testar conexão
            $pdo = new PDO('sqlite:' . $dbPath);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            return [
                'success' => true,
                'message' => 'Banco SQLite criado com sucesso!',
                'database_path' => $dbPath
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao criar banco SQLite: ' . $e->getMessage()
            ];
        }
    }
    
    private function createMySQLDatabase($config) {
        try {
            // Conectar sem especificar o banco
            $dsn = "mysql:host={$config['db_host']};port={$config['db_port']};charset=utf8mb4";
            $pdo = new PDO($dsn, $config['db_user'], $config['db_pass']);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            // Criar banco se não existir
            $dbName = $config['db_name'];
            $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$dbName}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            
            // Testar conexão com o banco criado
            $dsn = "mysql:host={$config['db_host']};port={$config['db_port']};dbname={$dbName};charset=utf8mb4";
            $pdo = new PDO($dsn, $config['db_user'], $config['db_pass']);
            
            return [
                'success' => true,
                'message' => 'Banco MySQL configurado com sucesso!',
                'database_name' => $dbName
            ];
            
        } catch (PDOException $e) {
            return [
                'success' => false,
                'message' => 'Erro ao configurar banco MySQL: ' . $e->getMessage(),
                'error_code' => $e->getCode()
            ];
        }
    }
    
    public function runMigrations() {
        try {
            // Verificar se o artisan existe
            if (!file_exists('../artisan')) {
                throw new Exception('Arquivo artisan não encontrado');
            }
            
            // Executar migrações
            $output = [];
            $returnCode = 0;
            
            exec('cd .. && php artisan migrate --force 2>&1', $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new Exception('Erro ao executar migrações: ' . implode("\n", $output));
            }
            
            return [
                'success' => true,
                'message' => 'Migrações executadas com sucesso!',
                'output' => $output
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    public function runSeeders() {
        try {
            $output = [];
            $returnCode = 0;
            
            // Executar seeder padrão
            exec('cd .. && php artisan db:seed --class=DefaultTemplateSeeder --force 2>&1', $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new Exception('Erro ao executar seeders: ' . implode("\n", $output));
            }
            
            return [
                'success' => true,
                'message' => 'Dados iniciais inseridos com sucesso!',
                'output' => $output
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    private function createConnection($config) {
        if ($config['db_type'] === 'sqlite') {
            $dbPath = '../database/database.sqlite';
            $pdo = new PDO('sqlite:' . $dbPath);
        } else {
            $dsn = "mysql:host={$config['db_host']};port={$config['db_port']};dbname={$config['db_name']};charset=utf8mb4";
            $pdo = new PDO($dsn, $config['db_user'], $config['db_pass']);
        }
        
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return $pdo;
    }
    
    public function createAdminUser($config, $dbConfig) {
        try {
            $pdo = $this->createConnection($dbConfig);
            
            // Verificar se já existe um usuário admin
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE email = ?");
            $stmt->execute([$config['admin_email']]);
            
            if ($stmt->fetchColumn() > 0) {
                return [
                    'success' => false,
                    'message' => 'Já existe um usuário com este email'
                ];
            }
            
            // Criar usuário admin
            $hashedPassword = password_hash($config['admin_password'], PASSWORD_DEFAULT);
            
            $stmt = $pdo->prepare("
                INSERT INTO users (name, email, password, email_verified_at, created_at, updated_at) 
                VALUES (?, ?, ?, NOW(), NOW(), NOW())
            ");
            
            $stmt->execute([
                'Administrador',
                $config['admin_email'],
                $hashedPassword
            ]);
            
            return [
                'success' => true,
                'message' => 'Usuário administrador criado com sucesso!',
                'user_id' => $pdo->lastInsertId()
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Erro ao criar usuário admin: ' . $e->getMessage()
            ];
        }
    }
}

// Processar requisições
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $installer = new DatabaseInstaller();
    
    switch ($action) {
        case 'test_connection':
            echo json_encode($installer->testConnection($_POST));
            break;
            
        case 'create_database':
            echo json_encode($installer->createDatabase($_POST));
            break;
            
        case 'run_migrations':
            echo json_encode($installer->runMigrations());
            break;
            
        case 'run_seeders':
            echo json_encode($installer->runSeeders());
            break;
            
        case 'create_admin':
            $dbConfig = [
                'db_type' => $_POST['db_type'],
                'db_host' => $_POST['db_host'] ?? '',
                'db_port' => $_POST['db_port'] ?? '',
                'db_name' => $_POST['db_name'] ?? '',
                'db_user' => $_POST['db_user'] ?? '',
                'db_pass' => $_POST['db_pass'] ?? ''
            ];
            echo json_encode($installer->createAdminUser($_POST, $dbConfig));
            break;
            
        default:
            echo json_encode([
                'success' => false,
                'message' => 'Ação inválida'
            ]);
    }
} else {
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido'
    ]);
}
?>

