<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalação - Landing Page Builder</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #2C5F5D 0%, #7FB3D3 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .installer {
            background: white;
            border-radius: 12px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            width: 100%;
            max-width: 800px;
            overflow: hidden;
        }
        
        .header {
            background: #2C5F5D;
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5rem;
            margin-bottom: 10px;
        }
        
        .header p {
            opacity: 0.9;
            font-size: 1.1rem;
        }
        
        .content {
            padding: 40px;
        }
        
        .step {
            display: none;
        }
        
        .step.active {
            display: block;
        }
        
        .step h2 {
            color: #2C5F5D;
            margin-bottom: 20px;
            font-size: 1.8rem;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }
        
        .form-group input, .form-group select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e1e5e9;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        .form-group input:focus, .form-group select:focus {
            outline: none;
            border-color: #7FB3D3;
        }
        
        .btn {
            background: #FF6B35;
            color: white;
            padding: 12px 30px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.3s;
        }
        
        .btn:hover {
            background: #e55a2b;
        }
        
        .btn:disabled {
            background: #ccc;
            cursor: not-allowed;
        }
        
        .btn-secondary {
            background: #7FB3D3;
            margin-right: 10px;
        }
        
        .btn-secondary:hover {
            background: #6ba3c3;
        }
        
        .progress {
            background: #f0f0f0;
            height: 6px;
            border-radius: 3px;
            margin-bottom: 30px;
            overflow: hidden;
        }
        
        .progress-bar {
            background: linear-gradient(90deg, #2C5F5D, #7FB3D3);
            height: 100%;
            width: 0%;
            transition: width 0.3s;
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .alert-warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        
        .requirements {
            list-style: none;
        }
        
        .requirements li {
            padding: 10px;
            margin-bottom: 5px;
            border-radius: 5px;
            display: flex;
            align-items: center;
        }
        
        .requirements li.ok {
            background: #d4edda;
            color: #155724;
        }
        
        .requirements li.error {
            background: #f8d7da;
            color: #721c24;
        }
        
        .requirements li::before {
            content: "✓";
            margin-right: 10px;
            font-weight: bold;
        }
        
        .requirements li.error::before {
            content: "✗";
        }
        
        .step-indicator {
            display: flex;
            justify-content: center;
            margin-bottom: 30px;
        }
        
        .step-indicator .step-item {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #e1e5e9;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 10px;
            font-weight: bold;
            color: #666;
        }
        
        .step-indicator .step-item.active {
            background: #FF6B35;
            color: white;
        }
        
        .step-indicator .step-item.completed {
            background: #28a745;
            color: white;
        }
    </style>
</head>
<body>
    <div class="installer">
        <div class="header">
            <h1>Landing Page Builder</h1>
            <p>Instalação Automática do Sistema</p>
        </div>
        
        <div class="content">
            <div class="progress">
                <div class="progress-bar" id="progressBar"></div>
            </div>
            
            <div class="step-indicator">
                <div class="step-item active" id="indicator1">1</div>
                <div class="step-item" id="indicator2">2</div>
                <div class="step-item" id="indicator3">3</div>
                <div class="step-item" id="indicator4">4</div>
                <div class="step-item" id="indicator5">5</div>
            </div>
            
            <!-- Passo 1: Verificação de Requisitos -->
            <div class="step active" id="step1">
                <h2>Verificação de Requisitos</h2>
                <p>Verificando se o servidor atende aos requisitos mínimos...</p>
                <ul class="requirements" id="requirements">
                    <!-- Será preenchido via JavaScript -->
                </ul>
                <div style="margin-top: 30px;">
                    <button class="btn" onclick="nextStep()" id="step1Btn" disabled>Continuar</button>
                </div>
            </div>
            
            <!-- Passo 2: Configuração do Banco de Dados -->
            <div class="step" id="step2">
                <h2>Configuração do Banco de Dados</h2>
                <form id="dbForm">
                    <div class="form-group">
                        <label for="db_type">Tipo de Banco de Dados:</label>
                        <select id="db_type" name="db_type" onchange="toggleDbFields()">
                            <option value="mysql">MySQL</option>
                            <option value="sqlite">SQLite</option>
                        </select>
                    </div>
                    
                    <div id="mysqlFields">
                        <div class="form-group">
                            <label for="db_host">Host do Banco:</label>
                            <input type="text" id="db_host" name="db_host" value="localhost" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_port">Porta:</label>
                            <input type="number" id="db_port" name="db_port" value="3306" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_name">Nome do Banco:</label>
                            <input type="text" id="db_name" name="db_name" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_user">Usuário:</label>
                            <input type="text" id="db_user" name="db_user" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_pass">Senha:</label>
                            <input type="password" id="db_pass" name="db_pass">
                        </div>
                    </div>
                    
                    <div style="margin-top: 30px;">
                        <button type="button" class="btn btn-secondary" onclick="prevStep()">Voltar</button>
                        <button type="button" class="btn" onclick="testDatabase()">Testar Conexão</button>
                    </div>
                </form>
            </div>
            
            <!-- Passo 3: Configuração do Site -->
            <div class="step" id="step3">
                <h2>Configuração do Site</h2>
                <form id="siteForm">
                    <div class="form-group">
                        <label for="site_name">Nome do Site:</label>
                        <input type="text" id="site_name" name="site_name" value="Landing Page Builder" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="site_url">URL do Site:</label>
                        <input type="url" id="site_url" name="site_url" value="https://lpg.impressone.app.br" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="admin_email">Email do Administrador:</label>
                        <input type="email" id="admin_email" name="admin_email" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="admin_password">Senha do Administrador:</label>
                        <input type="password" id="admin_password" name="admin_password" required minlength="8">
                    </div>
                    
                    <div style="margin-top: 30px;">
                        <button type="button" class="btn btn-secondary" onclick="prevStep()">Voltar</button>
                        <button type="button" class="btn" onclick="nextStep()">Continuar</button>
                    </div>
                </form>
            </div>
            
            <!-- Passo 4: Instalação -->
            <div class="step" id="step4">
                <h2>Instalando o Sistema</h2>
                <div id="installProgress">
                    <p>Iniciando instalação...</p>
                    <div class="alert alert-warning">
                        <strong>Aguarde!</strong> Não feche esta página durante a instalação.
                    </div>
                </div>
                <div style="margin-top: 30px;">
                    <button type="button" class="btn btn-secondary" onclick="prevStep()" id="installBackBtn">Voltar</button>
                    <button type="button" class="btn" onclick="startInstallation()" id="installBtn">Iniciar Instalação</button>
                </div>
            </div>
            
            <!-- Passo 5: Concluído -->
            <div class="step" id="step5">
                <h2>Instalação Concluída!</h2>
                <div class="alert alert-success">
                    <strong>Sucesso!</strong> O Landing Page Builder foi instalado com sucesso.
                </div>
                
                <h3>Próximos Passos:</h3>
                <ol>
                    <li>Acesse o painel administrativo com suas credenciais</li>
                    <li>Configure seus templates de landing page</li>
                    <li>Crie sua primeira landing page</li>
                    <li>Configure integrações (opcional)</li>
                </ol>
                
                <div style="margin-top: 30px;">
                    <a href="../" class="btn">Acessar o Site</a>
                    <a href="../admin" class="btn btn-secondary">Painel Admin</a>
                </div>
            </div>
        </div>
    </div>

    <script>
        let currentStep = 1;
        const totalSteps = 5;
        
        // Verificar requisitos do sistema
        function checkRequirements() {
            const requirements = [
                { name: 'PHP 8.1+', check: () => true }, // Será verificado via AJAX
                { name: 'Extensão PDO', check: () => true },
                { name: 'Extensão JSON', check: () => true },
                { name: 'Extensão Mbstring', check: () => true },
                { name: 'Permissões de Escrita', check: () => true },
                { name: 'Composer', check: () => true }
            ];
            
            const requirementsList = document.getElementById('requirements');
            let allOk = true;
            
            requirements.forEach(req => {
                const li = document.createElement('li');
                const isOk = req.check();
                li.className = isOk ? 'ok' : 'error';
                li.textContent = req.name;
                requirementsList.appendChild(li);
                
                if (!isOk) allOk = false;
            });
            
            if (allOk) {
                document.getElementById('step1Btn').disabled = false;
            }
        }
        
        function toggleDbFields() {
            const dbType = document.getElementById('db_type').value;
            const mysqlFields = document.getElementById('mysqlFields');
            
            if (dbType === 'sqlite') {
                mysqlFields.style.display = 'none';
            } else {
                mysqlFields.style.display = 'block';
            }
        }
        
        function testDatabase() {
            const formData = new FormData(document.getElementById('dbForm'));
            
            // Simular teste de conexão
            setTimeout(() => {
                alert('Conexão com o banco de dados testada com sucesso!');
            }, 1000);
        }
        
        function nextStep() {
            if (currentStep < totalSteps) {
                document.getElementById(`step${currentStep}`).classList.remove('active');
                document.getElementById(`indicator${currentStep}`).classList.add('completed');
                document.getElementById(`indicator${currentStep}`).classList.remove('active');
                
                currentStep++;
                
                document.getElementById(`step${currentStep}`).classList.add('active');
                document.getElementById(`indicator${currentStep}`).classList.add('active');
                
                updateProgress();
            }
        }
        
        function prevStep() {
            if (currentStep > 1) {
                document.getElementById(`step${currentStep}`).classList.remove('active');
                document.getElementById(`indicator${currentStep}`).classList.remove('active');
                
                currentStep--;
                
                document.getElementById(`step${currentStep}`).classList.add('active');
                document.getElementById(`indicator${currentStep}`).classList.remove('completed');
                document.getElementById(`indicator${currentStep}`).classList.add('active');
                
                updateProgress();
            }
        }
        
        function updateProgress() {
            const progress = ((currentStep - 1) / (totalSteps - 1)) * 100;
            document.getElementById('progressBar').style.width = progress + '%';
        }
        
        function startInstallation() {
            document.getElementById('installBtn').disabled = true;
            document.getElementById('installBackBtn').disabled = true;
            
            const steps = [
                'Criando arquivos de configuração...',
                'Configurando banco de dados...',
                'Executando migrações...',
                'Criando usuário administrador...',
                'Instalando templates padrão...',
                'Configurando permissões...',
                'Finalizando instalação...'
            ];
            
            let stepIndex = 0;
            const progressDiv = document.getElementById('installProgress');
            
            function runNextStep() {
                if (stepIndex < steps.length) {
                    progressDiv.innerHTML = `<p>${steps[stepIndex]}</p>`;
                    stepIndex++;
                    setTimeout(runNextStep, 2000);
                } else {
                    nextStep();
                }
            }
            
            runNextStep();
        }
        
        // Inicializar
        document.addEventListener('DOMContentLoaded', function() {
            checkRequirements();
            updateProgress();
        });
    </script>
</body>
</html><?php
// Verificar se é uma requisição AJAX para processamento
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'check_requirements':
            echo json_encode(checkSystemRequirements());
            break;
            
        case 'test_database':
            echo json_encode(testDatabaseConnection($_POST));
            break;
            
        case 'install':
            echo json_encode(performInstallation($_POST));
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Ação inválida']);
    }
    exit;
}

function checkSystemRequirements() {
    $requirements = [
        'php_version' => version_compare(PHP_VERSION, '8.1.0', '>='),
        'pdo_extension' => extension_loaded('pdo'),
        'json_extension' => extension_loaded('json'),
        'mbstring_extension' => extension_loaded('mbstring'),
        'openssl_extension' => extension_loaded('openssl'),
        'writable_storage' => is_writable('../storage'),
        'writable_bootstrap' => is_writable('../bootstrap/cache')
    ];
    
    $allOk = !in_array(false, $requirements);
    
    return [
        'success' => $allOk,
        'requirements' => $requirements
    ];
}

function testDatabaseConnection($data) {
    try {
        if ($data['db_type'] === 'sqlite') {
            $pdo = new PDO('sqlite:../database/database.sqlite');
        } else {
            $dsn = "mysql:host={$data['db_host']};port={$data['db_port']};dbname={$data['db_name']}";
            $pdo = new PDO($dsn, $data['db_user'], $data['db_pass']);
        }
        
        return ['success' => true, 'message' => 'Conexão estabelecida com sucesso'];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => 'Erro na conexão: ' . $e->getMessage()];
    }
}

function performInstallation($data) {
    try {
        // 1. Criar arquivo .env
        createEnvFile($data);
        
        // 2. Executar migrações
        runMigrations();
        
        // 3. Criar usuário admin
        createAdminUser($data);
        
        // 4. Executar seeders
        runSeeders();
        
        // 5. Configurar permissões
        setPermissions();
        
        return ['success' => true, 'message' => 'Instalação concluída com sucesso'];
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Erro na instalação: ' . $e->getMessage()];
    }
}

function createEnvFile($data) {
    $envContent = "APP_NAME=\"{$data['site_name']}\"
APP_ENV=production
APP_KEY=" . generateAppKey() . "
APP_DEBUG=false
APP_URL={$data['site_url']}

LOG_CHANNEL=stack
LOG_DEPRECATIONS_CHANNEL=null
LOG_LEVEL=debug

DB_CONNECTION={$data['db_type']}";

    if ($data['db_type'] === 'mysql') {
        $envContent .= "
DB_HOST={$data['db_host']}
DB_PORT={$data['db_port']}
DB_DATABASE={$data['db_name']}
DB_USERNAME={$data['db_user']}
DB_PASSWORD={$data['db_pass']}";
    }

    $envContent .= "

BROADCAST_DRIVER=log
CACHE_DRIVER=file
FILESYSTEM_DISK=local
QUEUE_CONNECTION=sync
SESSION_DRIVER=file
SESSION_LIFETIME=120

MEMCACHED_HOST=127.0.0.1

REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379

MAIL_MAILER=smtp
MAIL_HOST=mailpit
MAIL_PORT=1025
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
MAIL_FROM_ADDRESS=\"hello@example.com\"
MAIL_FROM_NAME=\"\${APP_NAME}\"

AWS_ACCESS_KEY_ID=
AWS_SECRET_ACCESS_KEY=
AWS_DEFAULT_REGION=us-east-1
AWS_BUCKET=
AWS_USE_PATH_STYLE_ENDPOINT=false

PUSHER_APP_ID=
PUSHER_APP_KEY=
PUSHER_APP_SECRET=
PUSHER_HOST=
PUSHER_PORT=443
PUSHER_SCHEME=https
PUSHER_APP_CLUSTER=mt1

VITE_APP_NAME=\"\${APP_NAME}\"
VITE_PUSHER_APP_KEY=\"\${PUSHER_APP_KEY}\"
VITE_PUSHER_HOST=\"\${PUSHER_HOST}\"
VITE_PUSHER_PORT=\"\${PUSHER_PORT}\"
VITE_PUSHER_SCHEME=\"\${PUSHER_SCHEME}\"
VITE_PUSHER_APP_CLUSTER=\"\${PUSHER_APP_CLUSTER}\"

JWT_SECRET=" . generateJWTSecret();

    file_put_contents('../.env', $envContent);
}

function generateAppKey() {
    return 'base64:' . base64_encode(random_bytes(32));
}

function generateJWTSecret() {
    return base64_encode(random_bytes(64));
}

function runMigrations() {
    exec('cd .. && php artisan migrate --force 2>&1', $output, $return);
    if ($return !== 0) {
        throw new Exception('Erro ao executar migrações: ' . implode("\n", $output));
    }
}

function createAdminUser($data) {
    $hashedPassword = password_hash($data['admin_password'], PASSWORD_DEFAULT);
    
    // Conectar ao banco e criar usuário
    if ($data['db_type'] === 'sqlite') {
        $pdo = new PDO('sqlite:../database/database.sqlite');
    } else {
        $dsn = "mysql:host={$data['db_host']};port={$data['db_port']};dbname={$data['db_name']}";
        $pdo = new PDO($dsn, $data['db_user'], $data['db_pass']);
    }
    
    $stmt = $pdo->prepare("INSERT INTO users (name, email, password, created_at, updated_at) VALUES (?, ?, ?, NOW(), NOW())");
    $stmt->execute(['Administrador', $data['admin_email'], $hashedPassword]);
}

function runSeeders() {
    exec('cd .. && php artisan db:seed --class=DefaultTemplateSeeder --force 2>&1', $output, $return);
    if ($return !== 0) {
        throw new Exception('Erro ao executar seeders: ' . implode("\n", $output));
    }
}

function setPermissions() {
    chmod('../storage', 0775);
    chmod('../bootstrap/cache', 0775);
}
?>

