class LandingPageInstaller {
    constructor() {
        this.currentStep = 1;
        this.totalSteps = 5;
        this.requirements = {};
        this.dbConfig = {};
        this.siteConfig = {};
        
        this.init();
    }
    
    init() {
        this.updateProgress();
        this.checkRequirements();
        this.bindEvents();
    }
    
    bindEvents() {
        // Bind form submissions
        document.getElementById('dbForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.testDatabase();
        });
        
        document.getElementById('siteForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.validateSiteConfig();
        });
        
        // Bind database type change
        document.getElementById('db_type').addEventListener('change', () => {
            this.toggleDbFields();
        });
    }
    
    async checkRequirements() {
        try {
            const response = await fetch('index.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=check_requirements'
            });
            
            const result = await response.json();
            this.displayRequirements(result);
            
        } catch (error) {
            console.error('Erro ao verificar requisitos:', error);
            this.displayRequirementsError();
        }
    }
    
    displayRequirements(result) {
        const requirementsList = document.getElementById('requirements');
        requirementsList.innerHTML = '';
        
        const requirements = [
            { key: 'php_version', name: 'PHP 8.1 ou superior' },
            { key: 'pdo_extension', name: 'Extensão PDO' },
            { key: 'json_extension', name: 'Extensão JSON' },
            { key: 'mbstring_extension', name: 'Extensão Mbstring' },
            { key: 'openssl_extension', name: 'Extensão OpenSSL' },
            { key: 'writable_storage', name: 'Pasta storage gravável' },
            { key: 'writable_bootstrap', name: 'Pasta bootstrap/cache gravável' }
        ];
        
        let allOk = true;
        
        requirements.forEach(req => {
            const li = document.createElement('li');
            const isOk = result.requirements[req.key];
            li.className = isOk ? 'ok' : 'error';
            li.textContent = req.name;
            requirementsList.appendChild(li);
            
            if (!isOk) allOk = false;
        });
        
        this.requirements = result.requirements;
        
        if (allOk) {
            document.getElementById('step1Btn').disabled = false;
            this.showAlert('success', 'Todos os requisitos foram atendidos!');
        } else {
            this.showAlert('error', 'Alguns requisitos não foram atendidos. Corrija-os antes de continuar.');
        }
    }
    
    displayRequirementsError() {
        const requirementsList = document.getElementById('requirements');
        requirementsList.innerHTML = '<li class="error">Erro ao verificar requisitos do sistema</li>';
        this.showAlert('error', 'Não foi possível verificar os requisitos do sistema.');
    }
    
    toggleDbFields() {
        const dbType = document.getElementById('db_type').value;
        const mysqlFields = document.getElementById('mysqlFields');
        
        if (dbType === 'sqlite') {
            mysqlFields.style.display = 'none';
            // Remove required attributes for MySQL fields
            mysqlFields.querySelectorAll('input[required]').forEach(input => {
                input.removeAttribute('required');
            });
        } else {
            mysqlFields.style.display = 'block';
            // Add required attributes back
            mysqlFields.querySelectorAll('input').forEach(input => {
                if (input.id !== 'db_pass') { // Password can be empty
                    input.setAttribute('required', 'required');
                }
            });
        }
    }
    
    async testDatabase() {
        const formData = new FormData(document.getElementById('dbForm'));
        formData.append('action', 'test_database');
        
        try {
            const response = await fetch('index.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showAlert('success', result.message);
                this.dbConfig = Object.fromEntries(formData);
                setTimeout(() => this.nextStep(), 1500);
            } else {
                this.showAlert('error', result.message);
            }
            
        } catch (error) {
            console.error('Erro ao testar banco:', error);
            this.showAlert('error', 'Erro ao testar conexão com o banco de dados.');
        }
    }
    
    validateSiteConfig() {
        const form = document.getElementById('siteForm');
        const formData = new FormData(form);
        
        // Validações básicas
        const siteUrl = formData.get('site_url');
        const adminEmail = formData.get('admin_email');
        const adminPassword = formData.get('admin_password');
        
        if (!this.isValidUrl(siteUrl)) {
            this.showAlert('error', 'URL do site inválida.');
            return;
        }
        
        if (!this.isValidEmail(adminEmail)) {
            this.showAlert('error', 'Email do administrador inválido.');
            return;
        }
        
        if (adminPassword.length < 8) {
            this.showAlert('error', 'A senha deve ter pelo menos 8 caracteres.');
            return;
        }
        
        this.siteConfig = Object.fromEntries(formData);
        this.nextStep();
    }
    
    isValidUrl(string) {
        try {
            new URL(string);
            return true;
        } catch (_) {
            return false;
        }
    }
    
    isValidEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
    
    async startInstallation() {
        document.getElementById('installBtn').disabled = true;
        document.getElementById('installBackBtn').disabled = true;
        
        const steps = [
            'Criando arquivos de configuração...',
            'Configurando banco de dados...',
            'Executando migrações...',
            'Criando usuário administrador...',
            'Instalando templates padrão...',
            'Configurando permissões...',
            'Finalizando instalação...'
        ];
        
        const progressDiv = document.getElementById('installProgress');
        
        for (let i = 0; i < steps.length; i++) {
            progressDiv.innerHTML = `<p>${steps[i]}</p>`;
            await this.sleep(1500);
        }
        
        // Executar instalação real
        try {
            const installData = { ...this.dbConfig, ...this.siteConfig };
            const formData = new FormData();
            
            Object.keys(installData).forEach(key => {
                formData.append(key, installData[key]);
            });
            formData.append('action', 'install');
            
            const response = await fetch('index.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                progressDiv.innerHTML = '<div class="alert alert-success">Instalação concluída com sucesso!</div>';
                setTimeout(() => this.nextStep(), 2000);
            } else {
                throw new Error(result.message);
            }
            
        } catch (error) {
            console.error('Erro na instalação:', error);
            progressDiv.innerHTML = `<div class="alert alert-error">Erro na instalação: ${error.message}</div>`;
            document.getElementById('installBackBtn').disabled = false;
        }
    }
    
    nextStep() {
        if (this.currentStep < this.totalSteps) {
            document.getElementById(`step${this.currentStep}`).classList.remove('active');
            document.getElementById(`indicator${this.currentStep}`).classList.add('completed');
            document.getElementById(`indicator${this.currentStep}`).classList.remove('active');
            
            this.currentStep++;
            
            document.getElementById(`step${this.currentStep}`).classList.add('active');
            document.getElementById(`indicator${this.currentStep}`).classList.add('active');
            
            this.updateProgress();
        }
    }
    
    prevStep() {
        if (this.currentStep > 1) {
            document.getElementById(`step${this.currentStep}`).classList.remove('active');
            document.getElementById(`indicator${this.currentStep}`).classList.remove('active');
            
            this.currentStep--;
            
            document.getElementById(`step${this.currentStep}`).classList.add('active');
            document.getElementById(`indicator${this.currentStep}`).classList.remove('completed');
            document.getElementById(`indicator${this.currentStep}`).classList.add('active');
            
            this.updateProgress();
        }
    }
    
    updateProgress() {
        const progress = ((this.currentStep - 1) / (this.totalSteps - 1)) * 100;
        document.getElementById('progressBar').style.width = progress + '%';
    }
    
    showAlert(type, message) {
        // Remove alertas existentes
        const existingAlerts = document.querySelectorAll('.alert');
        existingAlerts.forEach(alert => alert.remove());
        
        // Criar novo alerta
        const alert = document.createElement('div');
        alert.className = `alert alert-${type}`;
        alert.textContent = message;
        
        // Inserir no step atual
        const currentStepEl = document.getElementById(`step${this.currentStep}`);
        const firstChild = currentStepEl.querySelector('h2').nextElementSibling;
        currentStepEl.insertBefore(alert, firstChild);
        
        // Auto-remover alertas de sucesso
        if (type === 'success') {
            setTimeout(() => alert.remove(), 5000);
        }
    }
    
    sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
}

// Inicializar quando o DOM estiver carregado
document.addEventListener('DOMContentLoaded', function() {
    window.installer = new LandingPageInstaller();
});

// Funções globais para compatibilidade com HTML inline
function nextStep() {
    window.installer.nextStep();
}

function prevStep() {
    window.installer.prevStep();
}

function testDatabase() {
    window.installer.testDatabase();
}

function startInstallation() {
    window.installer.startInstallation();
}

function toggleDbFields() {
    window.installer.toggleDbFields();
}

